%% MAPIR Survey3 Orange, Cyan and Near Infraread
%
% by Martin H. Trauth, 12 August 2025
% http://mres.uni-potsdam.de
%
% Channel 1 = 615 nm (Orange)
% Channel 2 = 490 nm (Cyan or Blue/Green)
% Channel 3 = 808 nm (NIR or Near Infrared)

%%
clear, clc, close all

%%
% File name, file info and dimension of the image.
% Auto White Balance, 1/250 s shutter speed, center focus
filename = '2025_0812_101917_003.RAW';
inf = rawinfo(filename);
row = inf.CFAImageSize(:,1);
col = inf.CFAImageSize(:,2);

%%
% Load data from file.
id = fopen(filename);
I1 = fread(id,[col,row],'ubit12=>uint16');
I1 = I1';

%%
% We then demosaic the RGGB Bayer pattern encoded image to a truecolor
% (RGB) image using gradient-corrected linear interpolation.
I2 = demosaic(I1,'rggb');

%%
% The 16 bit image has a [6 490] range of values, the full 16 bit range is
% [0 65535] and therefore displaying the image with imshow yields a very
% dark image.
min(I2(:))
max(I2(:))
I3 = rescale(I2);

%%
% Mark rectangular white area
imshow(I3), title('Mark rectangular white area')
roi = drawrectangle;
position = roi.Position;
position = round(position);

%%
% Channel 1 = 615 nm (Orange)
B_white = mean2(I3(position(2)-position(4):position(2), ...
                   position(1):position(1)+position(3),1));
I4(:,:,1) = uint16(single(I3(:,:,1))*single(2^16)/B_white);

% Channel 2 = 490 nm (Cyan or Blue/Green)
B_white = mean2(I3(position(2)-position(4):position(2), ...
                   position(1):position(1)+position(3),2));
I4(:,:,2) = uint16(single(I3(:,:,2))*single(2^16)/B_white);

% Channel 3 = 808 nm (NIR or Near Infrared)
B_white = mean2(I3(position(2)-position(4):position(2), ...
                   position(1):position(1)+position(3),3));
I4(:,:,3) = uint16(single(I3(:,:,3))*single(2^16)/B_white);

%%
% Rectify, lick UL, LL, UR, LR.
imshow(I4), title('Click UL, LL, UR, LR, then press return')
movingpoints = ginput;

dx = (movingpoints(3,1)+movingpoints(4,1))/2- ...
     (movingpoints(1,1)+movingpoints(2,1))/2;
dy = (movingpoints(2,2)+movingpoints(4,2))/2- ...
     (movingpoints(1,2)+movingpoints(3,2))/2;

fixedpoints(1,:) = [0 0];
fixedpoints(2,:) = [0 dy];
fixedpoints(3,:) = [dx 0];
fixedpoints(4,:) = [dx dy];

tform = fitgeotrans(movingpoints,fixedpoints,'projective');

xLimitsIn = 0.5 + [0 size(I1,2)];
yLimitsIn = 0.5 + [0 size(I1,1)];

[XBounds,YBounds] = outputLimits(tform,xLimitsIn,yLimitsIn);
Rout = imref2d(size(I4),XBounds,YBounds);
I5 = imwarp(I4,tform,'OutputView',Rout);

tiledlayout(2,1)
nexttile, imshow(I4), title('Original Image')
nexttile, imshow(I5), title('Rectified Image')

%%
% Referencing. The rectangular area is 13.5 x 8.8 cm. We therefore mark
% measure the longer of the two sides of the rectangle.
clf
imshow(I5), title('Mark endpoints of the longer side, then press return')
[x,y] = ginput;

ix = 13.5*size(I5,2)/sqrt((y(2,1)-y(1,1))^2+(x(2,1)-x(1,1))^2);
iy = 13.5*size(I5,1)/sqrt((y(2,1)-y(1,1))^2+(x(2,1)-x(1,1))^2);

imshow(I4,...
    'XData',[0 ix],...
    'YData',[0 iy])
xlabel('Centimeters')
ylabel('Centimeters')
axis on
grid on

%%
% Crop, mark and drag, then double click into the image
imshow(I5)
title('Mark rectangular area, then double click to crop'), hold on
I6 = imcrop(I5);

tiledlayout(2,1)
nexttile, imshow(I5), title('Original Image')
nexttile, imshow(I6), title('Rectified and Cropped Image')

%%
% Image enhancement using adaptive histogram equilization
I7(:,:,1) = adapthisteq(I6(:,:,1));
I7(:,:,2) = adapthisteq(I6(:,:,2));
I7(:,:,3) = adapthisteq(I6(:,:,3));

%%
% Displaying images, store data
t = tiledlayout(2,2, ...
    'TileSpacing','compact');
nexttile, imshow(I3)
nexttile, imshow(I5)
nexttile, imshow(I6)
nexttile, imshow(I7)

print -dpng -r300 mapirsurvey3_615_490_808_1.png

%%
% Display Orange, Cyan and Near Infraread separately. No adapthisteq.
% Channel 1 = 615 nm (Orange)
% Channel 2 = 490 nm (Cyan or Blue/Green)
% Channel 3 = 808 nm (NIR or Near Infrared)
cmap = 'jet';
t = tiledlayout(2,2, ...
    'TileSpacing','compact');
nexttile, imshow(I6(:,:,1)), title('615 nm - Orange'), colormap(cmap)
nexttile, imshow(I6(:,:,2)), title('490 nm - Cyan'), colormap(cmap)
nexttile, imshow(I6(:,:,3)), title('808 nm - NIR'), colormap(cmap)
nexttile, imshow(I6)

print -dpng -r300 mapirsurvey3_615_490_808_2.png

%%
% Display Orange, Cyan and Near Infraread separately. With adapthisteq.
% Channel 1 = 615 nm (Orange)
% Channel 2 = 490 nm (Cyan or Blue/Green)
% Channel 3 = 808 nm (NIR or Near Infrared)
cmap = 'jet';
t = tiledlayout(2,2, ...
    'TileSpacing','compact');
nexttile, imshow(I7(:,:,1)), title('615 nm - Orange'), colormap(cmap)
nexttile, imshow(I7(:,:,2)), title('490 nm - Cyan'), colormap(cmap)
nexttile, imshow(I7(:,:,3)), title('808 nm - NIR'), colormap(cmap)
nexttile, imshow(I7)

print -dpng -r300 mapirsurvey3_615_490_808_3.png

close all

%%
mapirsurvey3_615_490_808 = I6;
save mapirsurvey3_615_490_808.mat mapirsurvey3_615_490_808