def canc(x = None,s = None,u = None,l = None,niter = None):
     
    '''CANC   Correlated Adaptive Noise Cancelling
    For two vectors x=signal+noise1 and s=signal+noise2, CANC extracts
    the correlated signal and removes the noise1 and noise2. The inputs
    are the two vectors x and s, the convergence factor u, the filter
    length l and the number of iterations iter. The output variables of
    CANC are the filtered primary signal zz, the extracted noise ee, the
    mean-squared error mer versus the number of performed iterations it
    with stepsize u, and the filter weights w for each data point in yn1
    and yn2.'''
    
#   References:
#   Hattingh, M. (1988): A new data adaptive filtering program to remove
#   noise from geophysical time- or space-series data. Computers and Geo-
#   sciences, 14: 467-480.
#   Haykin, S. (1991): Adaptive Filter Theory. Prentice-Hall, Englewood
#   Cliffs, New Jersey: 854.

    import numpy as np

    # Shape of s
    n = len(s)
    
    # Preallocating memory
    w  = np.zeros(l)
    
    xx  = np.zeros(n)
    yy  = np.zeros(n)
    ss  = np.zeros(n)
    
    zz  = np.zeros(n)
    ee  = np.zeros(n)
    
    mer = np.zeros(niter)
    ww  = np.zeros((n,l))

    y   = np.zeros(n)
    z   = np.zeros(n)
    e   = np.zeros(n)
    
    ors = s
    ms = np.mean(s)*np.ones(n)
    s = s - ms
    x = x - ms
    ors = ors - ms

    # Filter loop    
    for it in range(1,niter+1):
        for I in range(l+1,n+1):
            for k in range(1,l):
                xx[k] = x[I-k]
                ss[k] = s[I-k]
            
            for J in range(1,l):
                ww[I-1,J] = w[J]
                y[I-1] = y[I-1] + w[J]*xx[J]
                z[I-1] = z[I-1] + w[J]*ss[J]
                
            e[I-1] = ors[I-1-np.int64(np.fix(l/2))]-y[I-1]
            
            for J in range(1,l):
                w[J] = w[J]+2.*u*e[I-1]*xx[J]
        
        # Phase correction   
        for I in range(1,n-2):
            if I < np.int64(np.fix(l/2)):
                yy[I] = 0
                zz[I] = 0
                ee[I] = 0
                ww[I,:] = 0
                
            elif I > n-np.int64(np.fix(l/2)):
                yy[I] = 0
                zz[I] = 0
                ee[I] = 0
                ww[I,:] = 0
                
            else:
                yy[I] = y[I+np.int64(np.fix(l/2))]
                zz[I] = z[I+np.int64(np.fix(l/2))]
                ww[I,:] = ww[I+np.int64(np.fix(l/2))]
                ee[I] = np.abs(e[I+np.int64(np.fix(l/2))])
            
        y = np.zeros(n)
        z = np.zeros(n)
        
        mer[it-1] = np.mean(ee[np.int64(np.fix(l/2)) : \
                             n-np.int64(np.fix(l/2))]**2)
            
    
    return zz,ee,mer,ww

