% MATLAB script for wavelet power spectrum of the Chew Bahir K/Zr record
% 20 June 2022 - MH Trauth, U Potsdam
%
% Reference:
%
% Verena Foerster, Asfawossen Asrat, Christopher Bronk Ramsey, Erik T.
% Brown, Melissa S. Chapot, Alan Deino, Walter Duesing, Matthew Grove,
% Annette Hahn, Annett Junginger, Stefanie Kaboth-Bahr, Christine S. Lane,
% Stephan Opitz, Anders Noren, Helen M. Roberts, Mona Stockhecke, Ralph
% Tiedemann, Céline M. Vidal, Ralf Vogelsang, Andrew S. Cohen, Henry F.
% Lamb, Frank Schaebitz and Martin H. Trauth (2022) Pleistocene climate
% variability in eastern Africa influenced hominin evolution, Nature
% Geoscience, DOI: 10.1038/s41561-022-01032-y
clear, clc, close all

% Load data.
cpdata = load('script_foerster_ngeo_2022_data.txt');
t = cpdata(:,1);
y = cpdata(:,2);
y = (y-mean(y))./std(y);

% Compute wavelet power spectrum.
fb = cwtfilterbank('Wavelet','Morse',...
    'SamplingFrequency',1/0.1,...
    'SignalLength',length(y),...
    'WaveletParameters',[3,15]);
[wt,fr,coi] = cwt(y,'FilterBank',fb);

% Display wavelet power spectrum.
fig4 = figure('Position',[100 100 800 500],...
    'Color',[1 1 1]);
axes('Position',[0.1 0.55 0.8 0.3],...
    'Box','On',...
    'XLim',[0 650],...
    'XGrid','On',...
    'YDir','Reverse',...
    'FontSize',14,...
    'YGrid','On',...
    'LineWidth',1); hold on
line(t/1000,y,...
    'LineWidth',1);
ylabel('K/Zr');
b4 = axes('Position',[0.1 0.15 0.8 0.35],...
    'YLim',[0.005 0.2],...
    'XGrid','On',...
    'Color','None');
contourf(t/1000,fr,abs(wt),0:0.13:1.5)
caxis([0 1.2])
set(b4,...
    'XLim',[0 650],...
    'YLim',[0.005 0.2],...
    'YScale','log',...
    'FontSize',14,...
    'LineWidth',1)
b4.YTick = [0.01 0.025 0.05 0.1];
b4.YTickLabel = {'100','40','20','10'};
xlabel('Age (kyrs BP)');
ylabel('Period (kyrs)');
line(t/1000,coi,...
    'Color','w',...
    'LineStyle','--',...
    'LineWidth',1);
line(t/1000,0.01*ones(size(t)),...
    'Color','w',...
    'LineStyle',':',...
    'LineWidth',1);
line(t/1000,0.025*ones(size(t)),...
    'Color','w',...
    'LineStyle',':',...
    'LineWidth',1);
line(t/1000,0.05*ones(size(t)),...
    'Color','w',...
    'LineStyle',':',...
    'LineWidth',1);
c = colorbar(b4,'manual');
c.Label.String = 'Normalized Spectral Power';
pos = get(b4,'Position');
set(c,'Position',[pos(3)+.11 pos(2) .02 pos(4)],...
     'FontSize',14)
colormap((parula).^1.2)

% Save graphics.
orient(fig4,'landscape')
print(fig4,'extended_data_figure_4.pdf','-dpdf','-bestfit')
