% MATLAB script for change point analysis of the Chew Bahir K/Zr record
% 20 June 2022 - MH Trauth, U Potsdam
% 
% Reference:
%
% Verena Foerster, Asfawossen Asrat, Christopher Bronk Ramsey, Erik T.
% Brown, Melissa S. Chapot, Alan Deino, Walter Duesing, Matthew Grove,
% Annette Hahn, Annett Junginger, Stefanie Kaboth-Bahr, Christine S. Lane,
% Stephan Opitz, Anders Noren, Helen M. Roberts, Mona Stockhecke, Ralph
% Tiedemann, Céline M. Vidal, Ralf Vogelsang, Andrew S. Cohen, Henry F.
% Lamb, Frank Schaebitz and Martin H. Trauth (2022) Pleistocene climate
% variability in eastern Africa influenced hominin evolution, Nature
% Geoscience, DOI: 10.1038/s41561-022-01032-y
clear, clc, close all

% Load data
cpdata = load('script_foerster_ngeo_2022_data.txt');
t = cpdata(:,1);
y = cpdata(:,2);

% Changes in the measures of central tendency.
[ipoint,residual] = findchangepts(y,'MaxNumChanges',5);
m(1) = mean(y(1:ipoint(1)));
for i = 1 : length(ipoint)-1
    m(i+1) = mean(y(ipoint(i):ipoint(i+1)));
end
m(length(ipoint)+1) = mean(y(ipoint(i+1):end));

fig1 = figure('Position',[100 700 800 500],...
    'Color',[1 1 1]);
axes('Position',[0.1 0.55 0.8 0.3],...
    'Box','On',...
    'YDir','Reverse',...
    'XLim',[0 650],...
    'YLim',[0 50],...
    'XGrid','On',...
    'FontSize',14,...
    'YGrid','On',...
    'LineWidth',1); hold on
line(t/1000,y,'LineWidth',1);
line([0 t(ipoint(1))]/1000,[m(1) m(1)],...
    'Color',[0.8510 0.3255 0.0980],...
    'LineWidth',1)
for i = 1 : length(ipoint)-1
    line([t(ipoint(i)) t(ipoint(i+1))]/1000, ...
        [m(i+1) m(i+1)],...
    'Color',[0.8510 0.3255 0.0980],...
    'LineWidth',1)
end
line([t(ipoint(length(ipoint)))/1000 t(end)/1000], ...
    [m(end) m(end)],...
    'Color',[0.8510 0.3255 0.0980],...
    'LineWidth',1)
stem(t(ipoint)/1000,60*ones(size(t(ipoint))),...
    'BaseValue',-60,...
    'Color',[0.4667 0.6745 0.1882],...
    'Marker','none',...
    'LineWidth',1)
xlabel('Age (ka)')
ylabel('K/Zr')
text(620,45,'A',...
    'FontSize',24)

orient(fig1,'landscape')
print(fig1,'extended_data_figure_3a.pdf','-dpdf','-bestfit')

% Changes in the measures of dispersion.
[ipoint,residual] = findchangepts(y,...
    'MaxNumChanges',5,'Statistic','std');
s(1) = std(y(1:ipoint(1)));
for i = 1 : length(ipoint)-1
    s(i+1) = std(y(ipoint(i):ipoint(i+1)));
end
s(length(ipoint)+1) = std(y(ipoint(i+1):end));

fig2 = figure('Position',[100 400 800 500],...
    'Color',[1 1 1]);
axes('Position',[0.1 0.55 0.8 0.3],...
    'Color','None',...
    'Box','On',...
    'YDir','Reverse',...
    'XLim',[0 650],...
    'YLim',[0 50],...
    'FontSize',14,...
    'XGrid','On',...
    'YGrid','On',...
    'LineWidth',1); hold on
line(t/1000,y,...
    'LineWidth',1);
stem(t(ipoint)/1000,60*ones(size(t(ipoint))),...
    'BaseValue',-60,...
    'Color',[0.4667 0.6745 0.1882],...
    'Marker','none',...
    'LineWidth',1)
xlabel('Age (ka)')
ylabel('K/Zr')
axes('Position',[0.1 0.55 0.8 0.3],...
    'XGrid','On',...
    'YGrid','On',...
    'YColor',[0.8510 0.3255 0.0980],...
    'YAxisLocation','right',...
    'Color','None',...
    'FontSize',14,...
    'XLim',[0 650],...
    'YLim',[0 10],...
    'LineWidth',1); hold on
line([0 t(ipoint(1))]/1000,[s(1) s(1)],...
    'Color',[0.8510 0.3255 0.0980],...
    'LineWidth',1)
for i = 1 : length(ipoint)-1
    line([t(ipoint(i)) t(ipoint(i+1))]/1000, ...
        [s(i+1) s(i+1)],...
    'Color',[0.8510 0.3255 0.0980],...
    'LineWidth',1)
end
line([t(ipoint(length(ipoint))) t(end)]/1000, ...
    [s(end) s(end)],...
    'Color',[0.8510 0.3255 0.0980],...
    'LineWidth',1)
xlabel('Age (ka)')
ylabel('Standard Deviation')
text(620,45,'B',...
    'FontSize',24)

orient(fig2,'landscape')
print(fig2,'extended_data_figure_3b.pdf','-dpdf','-bestfit')

% Changes in the trend.
[ipoint,residual] = findchangepts(y,...
    'MaxNumChanges',5,'Statistic','linear');
p(1,:) = polyfit(t(1:ipoint(1)),y(1:ipoint(1)),1);
for i = 1 : length(ipoint)-1
    p(i+1,:) = polyfit(t(ipoint(i):ipoint(i+1)),...
        y(ipoint(i):ipoint(i+1)),1);
end
p(length(ipoint)+1,:) = polyfit(t(ipoint(length(ipoint)):end),...
    y(ipoint(length(ipoint)):end),1);

fig3 = figure('Position',[100 100 800 500],...
    'Color',[1 1 1]);
axes('Position',[0.1 0.55 0.8 0.3],...
    'Box','On',...
    'YDir','Reverse',...
    'YLim',[0 50],...
    'XLim',[0 650],...
    'XGrid','On',...
    'FontSize',14,...
    'YGrid','On',...
     'LineWidth',1); hold on
line(t/1000,y,'LineWidth',1);
line(t(1:ipoint(1))/1000,polyval(p(1,:), ...
    t(1:ipoint(1))),...
    'Color',[0.8510 0.3255 0.0980],...
    'LineWidth',1)
for i = 1 : length(ipoint)-1
    line(t(ipoint(i:i+1))/1000,polyval(p(i+1,:), ...
        t(ipoint(i:i+1))),...
    'Color',[0.8510 0.3255 0.0980],...
    'LineWidth',1)
end
line(t(ipoint(length(ipoint)):end)/1000, ...
    polyval(p(end,:),...
    t(ipoint(length(ipoint)):end)),...
    'Color',[0.8510 0.3255 0.0980],...
    'LineWidth',1)
stem(t(ipoint)/1000,60*ones(size(t(ipoint))),...
    'BaseValue',-60,...
    'Color',[0.4667 0.6745 0.1882],...
    'Marker','none',...
    'LineWidth',1)
xlabel('Age (ka)')
ylabel('K/Zr')
text(620,45,'C',...
    'FontSize',24)

orient(fig3,'landscape')
print(fig3,'extended_data_figure_3c.pdf','-dpdf','-bestfit')
