function [r,d] = rp(varargin)
%RP   Calculates a recurrence plot
%    R=RP(X,E,THRESH,NORM,ALG) calculates the recurrence plot R from an
%    embedding vector X and using the threshold E. X is a N-by-M matrix
%    corresponding to N time points and M embedding dimensions.
%
%    [R,D]=RP(...) outputs the recurrence plot R and the underlying
%    distance matrix D.
%
%    Optional arguments:
%         NORM - is a string setting the norm for distance calculation in
%                phasespace. Can be 'euc' for euclidian norm (default) or
%                'max' for maximum norm.
%       THRESH - is a string that specifies how the threshold epsilon will
%                be calculated. With 'fix' (default) the RP is computed
%                with a fixed threshold epsilon specified by the input
%                parameter E. With 'var' the RP is computed with a fixed 
%                threshold epsilon, which corresponds to the lower 'E'
%                quantile (specified by E) of the distance distribution of
%                all points in phasespace. With 'fan' the RP is computed
%                with a variable threshold resulting in a fixed amount of
%                nearest neighbours in phasespace, specified by the
%                fraction E of recurrence points.
%
%    Reference:
%         Marwan, N., Romano, M. C., Thiel, M., Kurths, J. (2007).
%         Recurrence plots for the analysis of complex systems. Physics
%         Reports, 438, 237-329. 
%         Kraemer, K. H., Donner, R. V., Heitzig, J., & Marwan, N. (2018).
%         Recurrence threshold selection for obtaining robust recurrence
%         characteristics in different embedding dimensions. Chaos, 28,
%         085720.
%
%    Example:
%         N = 300; % length of time series
%         x = .9*sin((1:N)*2*pi/70); % Exemplary time series.
%         xVec = embed(x,2,17); % Embed into 2 dimensions using delay 17.
%         R = rp(xVec,.1,'fix','max'); % Calculate RP using maximum norm
%                                      % and fixed threshold
%         imagesc(R)

% Copyright (c) 2016-2019
% Potsdam Institute for Climate Impact Research, Germany
% Institute of Geosciences, University of Potsdam, Germany
% Norbert Marwan, Hauke Kraemer
% http://www.pik-potsdam.de and http://www.geo.uni-potsdam.de
%
% This program is free software; you can redistribute it and/or modify it
% under the terms of the GNU General Public License as published by the
% Free Software Foundation; either version 2 of the License, or any later
% version.

%%
% Check input.
narginchk(1,5)
nargoutchk(0,2)

%%
% Set default values for input parameters.
methLib = {'euc','max'};
thresLib = {'fix','var','fan'};

meth = 'max';
thres = 'fix';
e = 1;

%%
% Get input arguments.
% Set norm.
if nargin > 3
    if isa(varargin{4},'char') & ismember(varargin{4},methLib)
        meth = lower(varargin{4});
    else
        warning(['Specified norm should be one of the following possible values:',...
           10,sprintf('''%s'' ',methLib{:})])
    end
end

% Set method for threshold calculation.
if nargin > 2
    if isa(varargin{3},'char') & ismember(varargin{3},thresLib)
        thres = lower(varargin{3});
    else
        warning(['Specified way of calculating threshold should be one of the following possible values:',...
            10,sprintf('''%s'' ',thresLib{:})])
    end
end

% Set threshold value.
if nargin > 1
    if isa(varargin{2},'double')
        e = varargin{2};
    else
        warning('Threshold has to be numeric.')
    end
end

% Embedding vector.
x = varargin{1};
N = size(x);
if N(1) < N(2)
   error('Embedding dimension is larger than the length of the vector. Please check!')
end

%%
% Calculate distance matrix D using MATLAB's pdist function.
switch meth
    case 'euc'
        d = squareform(pdist(x));
    otherwise
        d = squareform(pdist(x,'chebychev'));
end

%%
% Apply threshold to get the final RP.
if strcmp(thres,'fix')
    r = double(d < e);
elseif strcmp(thres,'var')
    epsilon = quantile(d(:),e);
    r = double(d < epsilon);
elseif strcmp(thres,'fan')
    q = quantile(d,e);
    thresholds = repmat(q,N(1),1);
    r = double(d<thresholds);
end
