%% STATISTICAL ANALYSIS OF THE ODP 967 DUST RECORD
%
% Analysis from
%
% Trauth, M.H., Larrasoana, J.C., Mudelsee, M., 2009, Trends, rhythms and
% events in Plio-Pleistocene African climate. Quaternary Science Reviews,
% 28, 399-411.
%
% Data from
%
% Larrasoana, J.C., Roberts, A.P., Rohling, E.J., Winklhofer, M.,
% Wehausen, R., 2003. Three million years of monsoon variability over the
% northern Sahara. Climate Dynamics 21, 689-698.
%
clear, close all
dust_967 = load('dust_967.txt');

%%
% Eliminate duplicate data
for i = 1:size(dust_967(:,9))-1
    if dust_967(i,9) == dust_967(i+1,9)
       dust_967(i,9) = NaN;
    end
end
dust_967(isnan(dust_967(:,9))==1,:) = [];

%%
% Interpolate data upon an evenly-spaced time vector
t = 8 : 2 : 5000;
y = interp1(dust_967(:,9),dust_967(:,11),t,'linear','extrap');

%%
% Define windows
w = 200;

%%
% Running Mann-Whitney-Wilcoxon U-test
for j = 1:length(w)
na = w(j);
nb = w(j);
for i = w(j)/2+1:length(y)-w(j)/2
    [p,h] = ranksum(y(i-w(j)/2:i-1),y(i+1:i+w(j)/2));
    mwreal(j,i) = p;

end
mwreal(j,1:w(j)/2) = mwreal(j,w(j)/2+1) * ones(1,w(j)/2);
mwreal(j,length(y)-w(j)/2+1:length(y)) = ...
        mwreal(j,length(y)-w(j)/2) * ones(1,w(j)/2);
end

%%
% Display the results of running Mann-Whitney-Wilcoxon U-test
subplot(2,1,1), plot(t,y), axis([0 5000 0 1])
subplot(2,1,2), plot(t,log(mwreal)), axis([0 5000 -80 0])

%%
% Running Ansari-Bradley test
for j = 1:length(w)
df1 = w(j) - 1;
df2 = w(j) - 1;
for i = w(j)/2+1:length(y)-w(j)/2
    [h,p] = ansaribradley(y(i-w(j)/2:i-1),y(i+1:i+w(j)/2));
    abreal(j,i) = p;
end
abreal(j,1:w(j)/2) = abreal(j,w(j)/2+1) * ones(1,w(j)/2);
abreal(j,length(y)-w(j)/2+1:length(y)) = ...
    abreal(j,length(y)-w(j)/2) * ones(1,w(j)/2);
end

%%
% Display the results of running Ansari-Bradley test
subplot(2,1,1), plot(t,y), axis([0 5000 0 1])
subplot(2,1,2), plot(t,log(abreal)), axis([0 5000 -40 0])

%%
% Running F-test to detect transitions in the 100, 40 and 20 kyr frequency
% bands.
[b100,a100] = butter(4,[0.005 0.015]*4);
[b040,a040] = butter(5,[0.02 0.03]*4);
[b020,a020] = butter(6,[0.04 0.06]*4);
y_100 = filtfilt(b100,a100,y);
y_040 = filtfilt(b040,a040,y);
y_020 = filtfilt(b020,a020,y);

%%
% Compute running F-test for 20, 40 and 100 kyr frequency bands.
for j = 1:length(w)
df1 = w(j) - 1;
df2 = w(j) - 1;
for i = w(j)/2+1:length(y_020)-w(j)/2
    [h,p] = ansaribradley(y_020(i-w(j)/2:i-1),y_020(i+1:i+w(j)/2));
    abreal_020(j,i) = p;
end
abreal_020(j,1:w(j)/2) = abreal_020(j,w(j)/2+1) * ones(1,w(j)/2);
abreal_020(j,length(y)-w(j)/2+1:length(y)) = ...
    abreal_020(j,length(y)-w(j)/2) * ones(1,w(j)/2);
end

for j = 1:length(w)
df1 = w(j) - 1;
df2 = w(j) - 1;
for i = w(j)/2+1:length(y_040)-w(j)/2
    [h,p] = ansaribradley(y_040(i-w(j)/2:i-1),y_040(i+1:i+w(j)/2));
    abreal_040(j,i) = p;
end
abreal_040(j,1:w(j)/2) = abreal_040(j,w(j)/2+1) * ones(1,w(j)/2);
abreal_040(j,length(y)-w(j)/2+1:length(y)) = ...
    abreal_040(j,length(y)-w(j)/2) * ones(1,w(j)/2);
end

for j = 1:length(w)
df1 = w(j) - 1;
df2 = w(j) - 1;
for i = w(j)/2+1:length(y_100)-w(j)/2
    [h,p] = ansaribradley(y_100(i-w(j)/2:i-1),y_100(i+1:i+w(j)/2));
    abreal_100(j,i) = p;
end
abreal_100(j,1:w(j)/2) = abreal_100(j,w(j)/2+1) * ones(1,w(j)/2);
abreal_100(j,length(y)-w(j)/2+1:length(y)) = ...
    abreal_100(j,length(y)-w(j)/2) * ones(1,w(j)/2);
end

%%
% All graphs in one figure window
subplot(6,1,1), plot(t,y_020)
subplot(6,1,2), plot(t,log(abreal_020)), axis([0 5000 -100 0])
subplot(6,1,3), plot(t,y_040)
subplot(6,1,4), plot(t,log(abreal_040)), axis([0 5000 -100 0])
subplot(6,1,5), plot(t,y_100)
subplot(6,1,6), plot(t,log(abreal_100)), axis([0 5000 -100 0])