%% STATISTICAL ANALYSIS OF ODP 659, 721/722 and 967 DUST/ISOTOPE RECORDS
%
% Script to generate Figure 7 of Trauth et al. (2009)
%
% Analysis from
%
% Trauth, M.H., Larrasoana, J.C., Mudelsee, M., 2009, Trends, rhythms and
% events in Plio-Pleistocene African climate. Quaternary Science Reviews,
% 28, 399-411.
%
% Data from
%
% deMenocal, P., Bloemendal, J., King, J., 1991. A rock-magnetic record of
% monsoonal dust deposition to the Arabian Sea: evidence for a shift in the
% mode of deposition at 2.4 Ma. Proceedings of the ODP Scientific Results
% 177, 389-407.
%
% Larrasoana, J.C., Roberts, A.P., Rohling, E.J., Winklhofer, M.,
% Wehausen, R., 2003. Three million years of monsoon variability over the
% northern Sahara. Climate Dynamics 21, 689-698.
%
% Tiedemann, R., Sarnthein, M., Shackleton, N.J., 1994. Astronomical
% timescale for the Pliocene Atlantic d18O and dust flux records of Ocean
% Drilling Program site 659. Paleoceanography 9, 619-638.
%
% Load data and pretreatment
clear, close all
dust_967 = load('dust_967.txt');
load insolation_laskar_2004.mat
break967 = load('BREAKFIT967.DAT');

%% MATLAB SCRIPT FOR FIGURE 7 - ODP 967 DUST
% Eliminate duplicate data
for i = 1:size(dust_967(:,9))-1
    if dust_967(i,9) == dust_967(i+1,9)
       dust_967(i,9) = NaN;
    end
end
dust_967(isnan(dust_967(:,9))==1,:) = [];

%%
% Interpolate data upon an evenly-spaced time vector
t = 8 : 2 : 5000;
y = interp1(dust_967(:,9),dust_967(:,11),t,'linear','extrap');
inso = interp1(insolation(:,1),insolation(:,2),t,'linear');

%%
% Filter data
[blow300,alow300] = butter(4,0.0033*4);
[blow150,alow150] = butter(4,0.0067*4);
[blow050,alow050] = butter(4,0.0200*4);
y_low300 = filtfilt(blow300,alow300,y);
y_low150 = filtfilt(blow150,alow150,y);
y_low050 = filtfilt(blow050,alow050,y);
inso_low300 = filtfilt(blow300,alow300,inso);
inso_low150 = filtfilt(blow150,alow150,inso);
inso_low050 = filtfilt(blow050,alow050,inso);

%%
% Display filter results
subplot(2,1,1)
  plot(t,y_low050,t,y_low300)
  hold on
  plot(dust_967(:,9),dust_967(:,11))
  plot(break967(:,1),break967(:,3))
  axis([0 5000 0 1])
subplot(2,1,2)
  plot(t,inso_low050,t,inso_low300)

%% MATLAB SCRIPT FOR FIGURE 7 - ODP 721 DUST
% Load data and pretreatment
clear, close all
dust_721 = load('dust_721.txt');
load insolation_laskar_2004.mat
break721 = load('BREAKFIT721.DAT');

%%
% Eliminate duplicate data
for i = 1:size(dust_721(:,9))-1
    if dust_721(i,9) == dust_721(i+1,9)
       dust_721(i,9) = NaN;
    end
end
dust_721(isnan(dust_721(:,9))==1,:) = [];

%%
% Interpolate data upon an evenly-spaced time vector
t = 8 : 2 : 5000;
y = interp1(dust_721(:,9),dust_721(:,11),t,'linear');
inso = interp1(insolation(:,1),insolation(:,2),t,'linear');

%%
% Filter data
[blow300,alow300] = butter(4,0.0033*4);
[blow150,alow150] = butter(4,0.0067*4);
[blow050,alow050] = butter(4,0.0200*4);
y_low300 = filtfilt(blow300,alow300,y);
y_low150 = filtfilt(blow150,alow150,y);
y_low050 = filtfilt(blow050,alow050,y);
inso_low300 = filtfilt(blow300,alow300,inso);
inso_low150 = filtfilt(blow150,alow150,inso);
inso_low050 = filtfilt(blow050,alow050,inso);

%%
% Display filter results
subplot(2,1,1)
  plot(t,y_low050,t,y_low300)
  hold on
  plot(dust_721(:,9),dust_721(:,11))
  plot(break721(:,1),break721(:,3))
  axis([0 5000 0 4])
subplot(2,1,2)
  plot(t,inso_low050,t,inso_low300)
  
%% MATLAB SCRIPT FOR FIGURE 7 - ODP 659 DUST
% Load data and pretreatment
clear, close all
dust_659 = load('dust_659.txt');
load insolation_laskar_2004.mat
break659 = load('BREAKFIT659.DAT');

%%
% Eliminate duplicate data
for i = 1:size(dust_659(:,9))-1
    if dust_659(i,9) == dust_659(i+1,9)
       dust_659(i,9) = NaN;
    end
end
dust_659(isnan(dust_659(:,9))==1,:) = [];

%%
% Interpolate data upon an evenly-spaced time vector
t = 8 : 2 : 5000;
y = interp1(dust_659(:,9),dust_659(:,11),t,'linear');
inso = interp1(insolation(:,1),insolation(:,2),t,'linear');

%%
% Filter data
[blow300,alow300] = butter(4,0.0033*4);
[blow150,alow150] = butter(4,0.0067*4);
[blow050,alow050] = butter(4,0.0200*4);
y_low300 = filtfilt(blow300,alow300,y);
y_low150 = filtfilt(blow150,alow150,y);
y_low050 = filtfilt(blow050,alow050,y);
inso_low300 = filtfilt(blow300,alow300,inso);
inso_low150 = filtfilt(blow150,alow150,inso);
inso_low050 = filtfilt(blow050,alow050,inso);

%%
% Display filter results
subplot(2,1,1)
  plot(t,y_low050,t,y_low300)
  hold on
  plot(dust_659(:,9),dust_659(:,11))
  plot(break659(:,1),break659(:,3))
  axis([0 5000 -0.5 2.5])
subplot(2,1,2)
  plot(t,inso_low050,t,inso_low300)

%% MATLAB SCRIPT FOR FIGURE 7 - ODP 659 ISOTOPE
% Load data and pretreatment
clear, close all
isotope_659 = load('isotope_659.txt');
load insolation_laskar_2004.mat

%%
% Eliminate duplicate data
for i = 1:size(isotope_659(:,9))-1
    if isotope_659(i,9) == isotope_659(i+1,9)
       isotope_659(i,9) = NaN;
    end
end
isotope_659(isnan(isotope_659(:,9))==1,:) = [];

%%
% Interpolate data upon an evenly-spaced time vector
t = 8 : 2 : 5000;
y = interp1(isotope_659(:,9),isotope_659(:,11),t,'linear');
inso = interp1(insolation(:,1),insolation(:,2),t,'linear');

%%
% Filter data
[blow300,alow300] = butter(4,0.0033*4);
[blow150,alow150] = butter(4,0.0067*4);
[blow050,alow050] = butter(4,0.0200*4);
y_low300 = filtfilt(blow300,alow300,y);
y_low150 = filtfilt(blow150,alow150,y);
y_low050 = filtfilt(blow050,alow050,y);
inso_low300 = filtfilt(blow300,alow300,inso);
inso_low150 = filtfilt(blow150,alow150,inso);
inso_low050 = filtfilt(blow050,alow050,inso);

%%
% Import break regression results for the full record
p = polyfit(t,y_low050,3);

%%
% Display filter results
subplot(2,1,1)
  plot(t,y_low050,t,y_low300,t,polyval(p,t))
  hold on
  plot(isotope_659(:,9),isotope_659(:,11))
  axis([0 5000 2 5])
subplot(2,1,2)
  plot(t,inso_low050,t,inso_low300)





