function [S,ABU] = bioturbation_1(L,C,M)
% [S,ABU] = BIOTURBATION_1(L,C,M) is a model for benthic mixing, where L is
% the number of sediment layers added to an initial pile of 10 layers, C
% is the number of particles per layer and M is the depth of homogeneous
% mixing. The function displays the mixed sediment column as colored
% pixels. It also returns the mixed sediment column as matrix and an array
% containing the number of particles per layer after mixing.
%
% Example: [s,abu] = bioturbation_1(30,70,5);

% Martin Trauth 13 April 2017

% Prepare video file
v = VideoWriter('bioturbation_1.avi');
v.FrameRate = 4;
v.Quality = 100;
open(v);

% Generate initial input matrices consisting of blue pixels
S = [zeros(10,C)];

% Generate an array of NaNs of the full size of sediment
N = NaN(10+L,C);

% Create figure window
figure('Position',[100 200 560 400],...
       'Color',[1 1 1]);
   
% Define colormap
cmap = [0 0.4470 0.7410;...
        0.9290 0.6940 0.1250;...
        0.8500 0.3250 0.0980];

% Loops for sediment layers
for i = 1:L

  % Sedimentation of one single layer of yellow pixels
  if i == 1
    S(size(S,1)+1,:)=ones(1,size(S,2));
  else
    S(size(S,1)+1,:)=zeros(1,size(S,2));
  end
  
  % Plotting the sediment layer before mixing
  if i == 1
      N(1:size(S,1),1:size(S,2)) = S;
      pcolor(N)
      colormap(cmap)
      axis equal, axis off
      title('Sediment Before Mixing','FontSize',14)
      pause(0.1)
  end
  
  drawnow
  GF = getframe(gcf);
  writeVideo(v,GF);
    
  % Mixing
  for n = 1:C
    % Compute random permutations of the integers from 1 to C
    z = randperm(M);
    % Store pixels to be mixed in second variable
    NS(1:M,n) = S(size(S,1)-M+1:size(S,1),n);
    % Mix pixels and store in original matrix S
    S(size(S,1)-M+1:size(S,1),n)=NS(z,n);
  end

  % Plotting after mixing the sediment layer.
  N(1:size(S,1),1:size(S,2)) = S;
  pcolor(N)
  colormap(cmap)
  axis equal, axis off
  if i == 1
      title(['Sediment After Loop ',num2str(i)],'FontSize',14)
      pause(0.1)
  else
      title(['Sediment After Loop ',num2str(i)],'FontSize',14)
      pause(0.1)
  end
  
  % Create animation
  drawnow
  GF = getframe(gcf);
  writeVideo(v,GF);
  
end

print -dpng -r300 bioturbation_1_sed_vs1.png

% Compute the sum of particles per layer after mixing
ABU = sum(S');

% Plot the total abundances down core
figure('Position',[700 200 560 400],...
       'Color',[1 1 1]);
axes('Box','On',...
    'LineWidth',1,...
    'FontSize',14)
line(L-ABU,1:length(ABU),...
    'Color',cmap(1,:),...
    'LineWidth',2)
line(ABU,1:length(ABU),...
    'Color',cmap(3,:),...
    'LineWidth',2)
title(['Sediment Abundances After Loop ',...
    num2str(i)],'FontSize',14)
xlabel('Absolute Abundance',...
    'FontSize',14)
ylabel('Depth in Core (cm)',...
    'FontSize',14)

% Close video file
close(v)

print -dpng -r300 bioturbation_1_abu_vs1.png


